part of utils;



abstract class Validator<T> {
  final String error;

  Validator(this.error);

  bool isValid(T value);

  String? call(T value) => isValid(value) ? null : error;

  bool hasMatch(
    String pattern,
    String value, {
    bool caseSensitive = true,
  }) =>
      RegExp(
        pattern,
        caseSensitive: caseSensitive,
      ).hasMatch(value);
}

class RequiredValidator extends Validator<String?> {
  RequiredValidator() : super('This field is required');

  @override
  bool isValid(String? value) => (value ?? '').isNotEmpty;
}

class EmailValidator extends Validator<String?> {
  final Pattern _pattern =
      r'^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$';

  EmailValidator() : super('Email is invalid');

  @override
  bool isValid(String? value) {
    if ((value ?? '').isEmpty) return true;
    return hasMatch(_pattern.toString(), value!, caseSensitive: false);
  }
}

class PasswordValidator extends Validator<String?> {
  final Pattern _pattern = r'(?![0-9]+$)(?![a-zA-Z]+$)[0-9A-Za-z]{6,16}$';

  PasswordValidator()
      : super('The password oil consists of 6 to 16 digits and letters');

  @override
  bool isValid(String? value) {
    if ((value ?? '').isEmpty) return true;
    return hasMatch(_pattern.toString(), value!, caseSensitive: false);
  }
}

class EqualValidator extends Validator<String?> {
  final TextEditingController input;

  EqualValidator(this.input) : super('Does not match the password');

  @override
  bool isValid(String? value) {
    if ((value ?? '').isEmpty) return true;
    return input.value.text == value;
  }
}

class AmountValidator extends Validator<String?> {
  final Pattern _pattern =
      r'(^[1-9]([0-9]+)?(\.[0-9]{1,2})?$)|(^(0){1}$)|(^[0-9]\.[0-9]([0-9])?$)';

  AmountValidator() : super('Amount is invalid');

  @override
  bool isValid(String? value) {
    if ((value ?? '').isEmpty) return true;
    return hasMatch(_pattern.toString(), value!, caseSensitive: false);
  }
}


class ValidatorTool {

  // 验证手机号
  static bool isValidPhoneNumber(String value) {
    RegExp phonePattern = RegExp(r'^1[3-9]\d{9}$');
    return phonePattern.hasMatch(value);
  }

  // 密码
  static bool isValidPassword(String value) {
    RegExp passwordPattern = RegExp(r'^((?=.*\d)(?=.*[A-Z])(?=.*[a-z]))|((?=.*\d)(?=.*[A-Z])(?=.*[\W_]))|((?=.*\d)(?=.*[a-z])(?=.*[\W_]))|((?=.*[A-Z])(?=.*[a-z])(?=.*[\W_]))[a-zA-Z\d\W_]{8,12}$');
    // 密码必须是数字、字母、特殊符号三种及以上8-12位的组合
    return passwordPattern.hasMatch(value);
  }

  static bool isValidCode(String value) {
    RegExp passwordPattern = RegExp(r'^\d{6}$');
    return passwordPattern.hasMatch(value);
  }
}


