import { DomEditor, SlateTransforms } from '@wangeditor/editor';
import { h } from 'snabbdom';

const withAudioNode = (editor) => {
  const { isInline, isVoid, normalizeNode } = editor;
  const newEditor = editor;

  newEditor.isInline = (elem) => {
    const type = DomEditor.getNodeType(elem);
    if (type === 'chapterAudio') return true; // 设置为 inline
    return isInline(elem);
  };

  newEditor.isVoid = (elem) => {
    const type = DomEditor.getNodeType(elem);
    if (type === 'chapterAudio') return true; // 设置为 void
    return isVoid(elem);
  };

  // 重新 normalize
  newEditor.normalizeNode = ([node, path]) => {
    const type = DomEditor.getNodeType(node)
    if (type !== 'chapterAudio') {
      // 未命中 chapterAudio ，执行默认的 normalizeNode
      return normalizeNode([node, path])
    }
    
    // editor 顶级 node
    const topLevelNodes = newEditor.children || []

    // 后面必须跟一个 p header blockquote（否则后面无法继续输入文字）
    const nextNode = topLevelNodes[path[0] + 1] || {}
    const nextNodeType = DomEditor.getNodeType(nextNode)
    if (
      nextNodeType !== 'paragraph' &&
      nextNodeType !== 'blockquote' &&
      !nextNodeType.startsWith('header')
    ) {
      // link-card node 后面不是 p 或 header ，则插入一个空 p
      const p = { type: 'paragraph', children: [{ text: '' }] }
      const insertPath = [path[0] + 1]
      SlateTransforms.insertNodes(newEditor, p, {
        at: insertPath, // 在 link-card 后面插入
      })
    }
  }

  return newEditor; // 返回 newEditor ，重要！！！
};

// 在编辑器中渲染新元素
// 定义 renderElem 函数
const renderAudio = (elem, children, editor) => {
  // 获取“附件”的数据，参考上文 myResume 数据结构
  const { title = '', audioUrl = '' } = elem;

  const imgNode = h(
    'span',
    {
      props: {
        className: 'chapter-audio-pic',
        style: { textAlign: 'center' }
      }
    },
    [h('audio', {props: { src: audioUrl, controls: true, autoPlay: true, width: 220, height: 100 } })]
  )
  const titleNode = h(
    'span',
    {
      props: {
        className: 'chapter-audio-title',
      },
      style: { textAlign: 'center', color: '#666', fontSize: '14px', margin: "0px", lineHeight: '20px' },
    },
    [title]
  )

  const element = [imgNode];
  // if (title) {
  //   element.push(titleNode);
  // }

  // 附件元素 vnode
  const attachVnode = h(
    // HTML tag
    'span',
    // HTML 属性、样式、事件
    {
      props: {  // HTML 属性，驼峰式写法
        contentEditable: false,
        className: 'chapter-audio-container',
      },
      style: { textAlign: 'center', padding: '10px 0px', }, // style ，驼峰式写法
      on: {
        click() {
          console.log('clicked');
        } /* 其他... */,
      },
    },
    // 子节点
    [...element],
  );

  return attachVnode;
};
const renderElemConf = {
  type: 'chapterAudio',
  renderElem: renderAudio,
};

// 把新元素转换为 HTML
const chapterAudioToHtml = (elem, childrenHtml) => {
  // 获取附件元素的数据
  const { title = '', audioUrl = '' } = elem;

  // 生成 HTML 代码
  const html = `<span class="chapter-audio-container" data-w-e-type="chapterAudio" data-w-e-is-void  data-audioUrl="${audioUrl}" data-title="${title}">
    <span class="chapter-audio-pic">
      <audio controls src="${audioUrl}" alt="${title}" />
    </span>
    
  </span>`;
  // <span class="chapter-audio-title">${title}</span>

  return html;
};
const chapterAudioElemToHtmlConf = {
  type: 'chapterAudio', // 新元素的 type ，重要！！！
  elemToHtml: chapterAudioToHtml,
};

// 解析新元素 HTML 到编辑器
const parseAudioHtml = (domElem, children, editor) => {

  // 从 DOM element 中获取“附件”的信息
  const audioUrl = domElem.getAttribute('data-audioUrl') || '';
  const title = domElem.getAttribute('data-title') || '';

  // 生成“附件”元素（按照此前约定的数据结构）
  const myResume = {
    type: 'chapterAudio',
    title,
    audioUrl,
    children: [{ text: '' }], // void node 必须有 children ，其中有一个空字符串，重要！！！
  };

  return myResume;
};
const parseAudioConf = {
  selector: 'span[data-w-e-type="chapterAudio"]', // CSS 选择器，匹配特定的 HTML 标签
  parseElemHtml: parseAudioHtml,
};

const chapterAudioModule = {
  editorPlugin: withAudioNode,
  renderElems: [renderElemConf],
  elemsToHtml: [chapterAudioElemToHtmlConf],
  parseElemsHtml: [parseAudioConf],
}

export default chapterAudioModule;
export {
  withAudioNode,
  renderElemConf,
  chapterAudioElemToHtmlConf,
  parseAudioConf,
};
