import { DomEditor, SlateTransforms, SlateEditor } from '@wangeditor/editor';
import { h } from 'snabbdom';
import iconClose from '@/assets/images/icon_chapter_close.png';
import { delChapterTopic } from '../utils/request';
import $ from 'jquery';
import '../utils/iconfont';
import { hexToRgb } from '@/utils/common';
import { findNodeWithParent } from '../utils/setting';

const withPracticeNode = (editor) => {
  const { isInline, isVoid, normalizeNode } = editor;
  const newEditor = editor;

  // newEditor.isInline = (elem) => {
  //   const type = DomEditor.getNodeType(elem);
  //   if (type === 'chapterPractice') return true; // 针对 type: attachment ，设置为 inline
  //   return isInline(elem);
  // };

  newEditor.isVoid = (elem) => {
    const type = DomEditor.getNodeType(elem);
    if (type === 'chapterPractice') return true; // 针对 type: attachment ，设置为 void
    return isVoid(elem);
  };

  // 重新 normalize
  newEditor.normalizeNode = ([node, path]) => {
    const type = DomEditor.getNodeType(node);
    if (type !== 'chapterPractice') {
      // 未命中 chapterPractice ，执行默认的 normalizeNode
      return normalizeNode([node, path]);
    }

    // editor 顶级 node
    const topLevelNodes = newEditor.children || [];

    // 后面必须跟一个 p header blockquote（否则后面无法继续输入文字）
    const nextNode = topLevelNodes[path[0] + 1] || {};
    const nextNodeType = DomEditor.getNodeType(nextNode);
    if (
      nextNodeType !== 'paragraph' &&
      nextNodeType !== 'blockquote' &&
      nextNodeType !== 'header' &&
      nextNodeType !== 'chapterPractice' &&
      nextNodeType !== 'chapterExpandRead' &&
      nextNodeType !== 'chapterSection' &&
      nextNodeType !== 'chapterHeader' &&
      nextNodeType !== 'chapterGallery' &&
      !nextNodeType.startsWith('header')
    ) {
      // link-card node 后面不是 p 或 header ，则插入一个空 p
      const p = { type: 'paragraph', children: [{ text: '' }] };
      const insertPath = [path[0] + 1];
      SlateTransforms.insertNodes(newEditor, p, {
        at: insertPath, // 在 link-card 后面插入
      });
    }
  };

  return newEditor; // 返回 newEditor ，重要！！！
};

// 在编辑器中渲染新元素
// 定义 renderElem 函数
const renderPractice = (elem, children, editor) => {
  const {
    title = '',
    practiceNum = '',
    callback = null,
    bookId = '',
    chapterId = '',
    theme = '#ab1941',
  } = elem;
  const str = `<svg class="svg-icon" style="color: #ffffff; fill: currentColor; overflow: hidden; width: 24px; height: 24px;" aria-hidden="true"><use xlink:href="#icon-writing"></use></svg>`;

  const bgColor = hexToRgb(theme);
  const bgColorStr = `rgba(${bgColor.r},${bgColor.g},${bgColor.b},0.1)`;

  const icon = h(
    'div',
    {
      props: {},
      style: {
        width: '50px',
        flex: '0 0 50px',
        textAlign: 'center',
        backgroundColor: `${theme}`,
      },
    },
    [
      h(
        'div',
        {
          props: { innerHTML: str },
          style: {
            padding: '10px 0',
            width: '100%',
            height: '24px',
            display: 'flex',
            alignItems: 'center',
            boxSizing: 'content-box',
            justifyContent: 'center',
          },
        },
        [],
      ),
    ],
  );

  const leftTitle = h(
    'div',
    {
      props: {
        contentEditable: false,
        className: `chapter-practice-title`,
      },
      style: {
        display: 'flex',
        flex: '1',
        cursor: 'pointer',
      },
    },
    [
      h(
        'div',
        {
          props: {},
          style: {
            flex: '1',
            whiteSpace: 'pre-wrap',
            wordBreak: 'break-all',
            wordWrap: 'break-word',
            fontSize: '18px',
            padding: '10px 20px',
            lineHeight: '24px',
          },
        },
        [title],
      ),
      h(
        'div',
        {
          props: {
            contentEditable: false,
            className: `chapter-practice-close`,
          },
          style: {
            flex: '0 0 60px',
            padding: '10px 0',
            textAlign: 'center',
          },
          on: {
            async click(ev) {
              ev.stopPropagation();
              ev.preventDefault();

              // 删除当前节点
              await delChapterTopic({
                position: practiceNum,
                book_id: bookId,
                chapter_id: chapterId,
              });

              try {
                const path = findNodeWithParent(editor.children, 'chapterPractice', 'practiceNum', practiceNum);
                SlateTransforms.removeNodes(editor, { at: path.reverse() });
              } catch (e) {
                console.log(e);
              }
            },
          },
        },
        [
          h('img', {
            props: { src: iconClose, width: 18, height: 18 },
            style: { cursor: 'pointer' },
          }),
        ],
      ),
    ],
  );
  // 附件元素 vnode
  const attachVnode = h(
    // HTML tag
    'div',
    // HTML 属性、样式、事件
    {
      props: {
        // HTML 属性，驼峰式写法
        contentEditable: false,
        className: `chapter-practice practice-${practiceNum}`,
        id: `practice-${practiceNum}`,
      },
      style: {
        border: `1px solid ${theme}`,
        borderRadius: '6px',
        backgroundColor: bgColorStr,
        display: 'flex',
        overflow: 'hidden',
        alignItems: 'stretch',
      }, // style ，驼峰式写法
      dataset: {
        theme,
        random: practiceNum,
      },
      on: {
        click(ev) {
          ev.stopPropagation();
          ev.preventDefault();
          
          try {
            callback && callback(practiceNum, title);
          } catch (e) {}
          localStorage.setItem(
            'practiceNum',
            JSON.stringify({ practiceNum, title, theme, random: Math.random() }),
          );
        } /* 其他... */,
      },
    },
    // 子节点
    [icon, leftTitle],
  );

  return attachVnode;
};
const renderElemConf = {
  type: 'chapterPractice',
  renderElem: renderPractice,
};

// 把新元素转换为 HTML
const chapterPracticeToHtml = (elem, childrenHtml) => {
  // 获取附件元素的数据
  const { title = '', practiceNum = '', bookId = '', chapterId = '', theme = '#ab1941' } = elem;

  const bgColor = hexToRgb(theme);
  const bgColorStr = `rgba(${bgColor.r},${bgColor.g},${bgColor.b}, 0.1)`;

  const str = `<svg class="svg-icon" style="color: #ffffff; fill: currentColor; overflow: hidden; width: 24px; height: 24px;" aria-hidden="true"><use xlink:href="#icon-writing"></use></svg>`;
  // <img src="${iconClose}" width="18" height="18" style="cursor: pointer" />

  const strClose = `<svg class="svg-icon" style="color: ${theme}; fill: currentColor; overflow: hidden; width: 24px; height: 24px;" aria-hidden="true"><use xlink:href="#icon-tubiao1"></use></svg>`;

  // 生成 HTML 代码
  const html = `<div class="chapter-practice practice-${practiceNum}" id="practice-${practiceNum}" data-w-e-type="chapterPractice" data-title="${title}"  data-theme="${theme}" data-bookId="${bookId}" data-chapterId="${chapterId}" data-practiceNum="${practiceNum}" style="border: 1px solid ${theme}; border-radius: 6px; display: flex; overflow:hidden; background-color: ${bgColorStr}; align-items: stretch;">
    <div style="flex: 0 0 50px; width: 50px; text-align: center; background-color: ${theme};"><div style="padding: 10px 0; width: 100%; height: 24px; display: flex; align-items: center; box-sizing: content-box; justify-content: center;">${str}</div></div>
    <div class="chapter-practice-title" style="display: flex; flex: 1; cursor: pointer;">
      <div style="flex: 1; font-size: 18px; line-height: 24px; padding: 10px 20px; white-space: pre-wrap; word-break: break-all; word-wrap: break-word;">${title}</div>
      <div class="chapter-practice-close" style="flex: 0 0 40px; display: flex; justify-content: center; text-align: right; padding-top: 12px;">${strClose}</div>
    </div>
  </div>`;

  return html;
};
const chapterPracticeElemToHtmlConf = {
  type: 'chapterPractice', // 新元素的 type ，重要！！！
  elemToHtml: chapterPracticeToHtml,
};

// 解析新元素 HTML 到编辑器
const parsePracticeHtml = (domElem, children, editor) => {
  // 从 DOM element 中获取“附件”的信息
  const practiceNum = domElem.getAttribute('data-practiceNum') || '';
  const title = domElem.getAttribute('data-title') || '';
  const bookId = domElem.getAttribute('data-bookId') || '';
  const chapterId = domElem.getAttribute('data-chapterId') || '';
  const theme = domElem.getAttribute('data-theme') || '#ab1941';

  // 生成“附件”元素（按照此前约定的数据结构）
  const myResume = {
    type: 'chapterPractice',
    title,
    practiceNum,
    bookId,
    chapterId,
    theme,
    children: [{ text: '' }], // void node 必须有 children ，其中有一个空字符串，重要！！！
  };

  return myResume;
};
const parsePracticeConf = {
  selector: 'div[data-w-e-type="chapterPractice"]', // CSS 选择器，匹配特定的 HTML 标签
  parseElemHtml: parsePracticeHtml,
};

const chapterPracticeModule = {
  editorPlugin: withPracticeNode,
  renderElems: [renderElemConf],
  elemsToHtml: [chapterPracticeElemToHtmlConf],
  parseElemsHtml: [parsePracticeConf],
};

export default chapterPracticeModule;
export { withPracticeNode, renderElemConf, chapterPracticeElemToHtmlConf, parsePracticeConf };
