import { DomEditor, SlateTransforms } from '@wangeditor/editor';
import { h } from 'snabbdom';

const withVideoNode = (editor) => {
  const { isInline, isVoid, normalizeNode } = editor;
  const newEditor = editor;

  newEditor.isInline = (elem) => {
    const type = DomEditor.getNodeType(elem);
    if (type === 'chapterVideo') return true; // 设置为 inline
    return isInline(elem);
  };

  newEditor.isVoid = (elem) => {
    const type = DomEditor.getNodeType(elem);
    if (type === 'chapterVideo') return true; // 设置为 void
    return isVoid(elem);
  };

  // 重新 normalize
  newEditor.normalizeNode = ([node, path]) => {
    const type = DomEditor.getNodeType(node)
    if (type !== 'chapterVideo') {
      // 未命中 chapterVideo ，执行默认的 normalizeNode
      return normalizeNode([node, path])
    }
    
    // editor 顶级 node
    const topLevelNodes = newEditor.children || []

    // 后面必须跟一个 p header blockquote（否则后面无法继续输入文字）
    const nextNode = topLevelNodes[path[0] + 1] || {}
    const nextNodeType = DomEditor.getNodeType(nextNode)
    if (
      nextNodeType !== 'paragraph' &&
      nextNodeType !== 'blockquote' &&
      nextNodeType !== 'header' &&
      nextNodeType !== 'chapterPractice' &&
      nextNodeType !== 'chapterExpandRead' &&
      nextNodeType !== 'chapterSection' &&
      nextNodeType !== 'chapterHeader' &&
      nextNodeType !== 'chapterGallery' &&
      !nextNodeType.startsWith('header')
    ) {
      // link-card node 后面不是 p 或 header ，则插入一个空 p
      const p = { type: 'paragraph', children: [{ text: '' }] }
      const insertPath = [path[0] + 1]
      SlateTransforms.insertNodes(newEditor, p, {
        at: insertPath, // 在 link-card 后面插入
      })
    }
  }

  return newEditor; // 返回 newEditor ，重要！！！
};

// 在编辑器中渲染新元素
// 定义 renderElem 函数
const renderVideo = (elem, children, editor) => {
  // 获取“附件”的数据，参考上文 myResume 数据结构
  const { title = '', videoUrl = '' } = elem;

  const imgNode = h(
    'span',
    {
      props: {
        className: 'chapter-video-pic',
      },
      style: { textAlign: 'center' }
    },
    [h('video', {props: { src: videoUrl, controls: true, autoPlay: true, width: 320, height: 100 } })]
  )
  const titleNode = h(
    'span',
    {
      props: {
        className: 'chapter-video-title',
      },
      style: { textAlign: 'center', color: '#666', fontSize: '14px', margin: "0px", lineHeight: '20px' },
    },
    [title]
  )

  const element = [imgNode];
  if (title) {
    // element.push(titleNode);
  }

  // 附件元素 vnode
  const attachVnode = h(
    // HTML tag
    'span',
    // HTML 属性、样式、事件
    {
      props: {  // HTML 属性，驼峰式写法
        contentEditable: false,
        className: 'chapter-video-container',
      },
      style: { textAlign: 'center', padding: '10px 0px', }, // style ，驼峰式写法
      on: {
        click() {
          console.log('clicked');
        } /* 其他... */,
      },
    },
    // 子节点
    [...element],
  );

  return attachVnode;
};
const renderElemConf = {
  type: 'chapterVideo',
  renderElem: renderVideo,
};

// 把新元素转换为 HTML
const chapterVideoToHtml = (elem, childrenHtml) => {
  // 获取附件元素的数据
  const { title = '', videoUrl = '' } = elem;

  // 生成 HTML 代码
  const html = `<span
    class="chapter-video-container"
    data-w-e-type="chapterVideo"
    data-w-e-is-void
    data-videoUrl="${videoUrl}"
    data-title="${title}"
  >
    <span class="chapter-video-pic">
      <video controls src="${videoUrl}" width="320" height="100" alt="${title}" />
    </span>
    
  </span>`;
  // <span class="chapter-video-title">${title}</span>

  return html;
};
const chapterVideoElemToHtmlConf = {
  type: 'chapterVideo', // 新元素的 type ，重要！！！
  elemToHtml: chapterVideoToHtml,
};

// 解析新元素 HTML 到编辑器
const parseVideoHtml = (domElem, children, editor) => {

  // 从 DOM element 中获取“附件”的信息
  const videoUrl = domElem.getAttribute('data-videoUrl') || '';
  const title = domElem.getAttribute('data-title') || '';

  // 生成“附件”元素（按照此前约定的数据结构）
  const myResume = {
    type: 'chapterVideo',
    title,
    videoUrl,
    children: [{ text: '' }], // void node 必须有 children ，其中有一个空字符串，重要！！！
  };

  return myResume;
};
const parseVideoConf = {
  selector: 'span[data-w-e-type="chapterVideo"]', // CSS 选择器，匹配特定的 HTML 标签
  parseElemHtml: parseVideoHtml,
};

const chapterVideoModule = {
  editorPlugin: withVideoNode,
  renderElems: [renderElemConf],
  elemsToHtml: [chapterVideoElemToHtmlConf],
  parseElemsHtml: [parseVideoConf],
}

export default chapterVideoModule;
export {
  withVideoNode,
  renderElemConf,
  chapterVideoElemToHtmlConf,
  parseVideoConf,
};
