import Log from './logs.model.js'
import dayjs from 'dayjs'

export const create = async (logData) => {
  const log = new Log(logData)
  return log.save()
}

export const createBatch = async (logs) => {
  return Log.insertMany(logs, { ordered: false })
}

export const query = async (filters = {}, options = {}) => {
  const { page = 1, limit = 50, sort = { createdAt: -1 } } = options
  const skip = (page - 1) * limit

  const query = {}

  if (filters.type) query.type = filters.type
  if (filters.level) query.level = filters.level
  if (filters.appName) query.appName = filters.appName
  if (filters.userId) query.userId = filters.userId
  if (filters.startTime || filters.endTime) {
    query.createdAt = {}
    // createdAt 是字符串格式 YYYY-MM-DD HH:mm:ss，直接使用字符串比较
    if (filters.startTime) query.createdAt.$gte = filters.startTime
    if (filters.endTime) query.createdAt.$lte = filters.endTime
  }
  if (filters.keyword) {
    query.message = { $regex: filters.keyword, $options: 'i' }
  }

  const [list, total] = await Promise.all([
    Log.find(query).sort(sort).skip(skip).limit(limit).lean(),
    Log.countDocuments(query),
  ])

  return {
    list,
    pagination: {
      page,
      limit,
      total,
      pages: Math.ceil(total / limit),
    },
  }
}

// 获取今日开始时间（服务器本地时间，字符串格式 YYYY-MM-DD HH:mm:ss）
const getTodayStart = () => {
  return dayjs().startOf('day').format('YYYY-MM-DD HH:mm:ss')
}

// 获取昨日时间范围（服务器本地时间，字符串格式）
const getYesterdayRange = () => {
  const today = dayjs().startOf('day').format('YYYY-MM-DD HH:mm:ss')
  const yesterday = dayjs().subtract(1, 'day').startOf('day').format('YYYY-MM-DD HH:mm:ss')
  return { start: yesterday, end: today }
}

export const getStats = async (filters = {}) => {
  const todayStart = getTodayStart()
  const { start: yesterdayStart, end: yesterdayEnd } = getYesterdayRange()

  const baseMatch = {}
  if (filters.appName) baseMatch.appName = filters.appName

  // 并行查询所有统计数据
  const [todayTotal, yesterdayTotal, todayErrors, yesterdayErrors, todayApps, yesterdayApps, recentLogs] =
    await Promise.all([
      // 今日日志总数
      Log.countDocuments({ ...baseMatch, createdAt: { $gte: todayStart } }),
      // 昨日日志总数
      Log.countDocuments({ ...baseMatch, createdAt: { $gte: yesterdayStart, $lt: yesterdayEnd } }),
      // 今日错误日志
      Log.countDocuments({ ...baseMatch, createdAt: { $gte: todayStart }, level: { $in: ['error', 'fatal'] } }),
      // 昨日错误日志
      Log.countDocuments({
        ...baseMatch,
        createdAt: { $gte: yesterdayStart, $lt: yesterdayEnd },
        level: { $in: ['error', 'fatal'] },
      }),
      // 今日活跃应用数
      Log.distinct('appName', { ...baseMatch, createdAt: { $gte: todayStart } }),
      // 昨日活跃应用数
      Log.distinct('appName', { ...baseMatch, createdAt: { $gte: yesterdayStart, $lt: yesterdayEnd } }),
      // 最近日志
      Log.find(baseMatch).sort({ createdAt: -1 }).limit(10).lean(),
    ])

  // 计算变化率
  const calcChange = (today, yesterday) => {
    if (yesterday === 0) return today > 0 ? 100 : 0
    return Number((((today - yesterday) / yesterday) * 100).toFixed(1))
  }

  return {
    // 今日日志
    todayTotal: {
      count: todayTotal,
      change: calcChange(todayTotal, yesterdayTotal),
    },
    // 错误日志
    errorTotal: {
      count: todayErrors,
      change: calcChange(todayErrors, yesterdayErrors),
    },
    // 活跃应用
    activeApps: {
      count: todayApps.length,
      change: todayApps.length - yesterdayApps.length,
    },
    // 最近日志
    recentLogs,
  }
}
