import crypto from 'crypto'
import axios from 'axios'
import config from '../../config.js'

let ticketCache = { jsapi_ticket: '', noncestr: '', timestamp: '', expiresAt: 0 }
const CACHE_DURATION = 60 * 60 * 1000 // 1 hour

export const getAccessToken = async (appId) => {
  const secret = config.wechat.apps[appId]
  if (!secret) throw new Error(`WeChat app secret not found for appId: ${appId}`)

  const { data } = await axios.get('https://api.weixin.qq.com/cgi-bin/token', {
    params: { grant_type: 'client_credential', appid: appId, secret },
  })

  if (data.errcode) throw new Error(`Failed to get access token: ${data.errmsg}`)
  return data.access_token
}

export const getJsapiTicket = async (appId) => {
  const accessToken = await getAccessToken(appId)
  const { data } = await axios.get('https://api.weixin.qq.com/cgi-bin/ticket/getticket', {
    params: { type: 'jsapi', access_token: accessToken },
  })

  if (data.errcode !== 0) throw new Error(`Failed to get JSAPI ticket: ${data.errmsg}`)
  return data.ticket
}

export const getCachedTicket = async (appId) => {
  const now = Date.now()
  if (ticketCache.jsapi_ticket && ticketCache.expiresAt > now) return ticketCache

  const ticket = await getJsapiTicket(appId)
  ticketCache = {
    jsapi_ticket: ticket,
    noncestr: Math.random().toString(36).substr(2, 15),
    timestamp: Math.floor(Date.now() / 1000).toString(),
    expiresAt: now + CACHE_DURATION,
  }
  return ticketCache
}

export const generateSignature = (ticket, noncestr, timestamp, url) => {
  const str = `jsapi_ticket=${ticket}&noncestr=${noncestr}&timestamp=${timestamp}&url=${url}`
  return crypto.createHash('sha1').update(str).digest('hex')
}

export const getUserInfo = async (appId, code) => {
  const secret = config.wechat.apps[appId]
  if (!secret) throw new Error(`WeChat app secret not found for appId: ${appId}`)

  const { data: tokenData } = await axios.get('https://api.weixin.qq.com/sns/oauth2/access_token', {
    params: { appid: appId, secret, code, grant_type: 'authorization_code' },
  })

  if (tokenData.errcode) throw new Error(`Failed to get access token: ${tokenData.errmsg}`)

  const { data: userInfo } = await axios.get('https://api.weixin.qq.com/sns/userinfo', {
    params: { access_token: tokenData.access_token, openid: tokenData.openid, lang: 'zh_CN' },
  })

  if (userInfo.errcode) throw new Error(`Failed to get user info: ${userInfo.errmsg}`)
  return userInfo
}
