import { UAParser } from 'ua-parser-js'
import * as logsService from './logs.service.js'
import logger from '../../lib/logger.js'
import { success, error } from '../../lib/response.js'

// 解析 userAgent 获取设备信息
const parseDevice = (userAgent) => {
  if (!userAgent) return null

  const parser = new UAParser(userAgent)
  const device = parser.getDevice()
  const os = parser.getOS()
  const browser = parser.getBrowser()

  return {
    type: device.type || 'desktop', // mobile | tablet | desktop
    os: os.name && os.version ? `${os.name} ${os.version}` : os.name || null,
    browser: browser.name && browser.version ? `${browser.name} ${browser.version}` : browser.name || null,
  }
}

export const create = async (req, res, next) => {
  try {
    const userAgent = req.get('user-agent')
    const logData = {
      ...req.body,
      ip: req.ip,
      userAgent,
      device: parseDevice(userAgent),
    }

    const log = await logsService.create(logData)
    success(res, log)
  } catch (err) {
    logger.error({ err }, 'Failed to create log')
    next(err)
  }
}

export const createBatch = async (req, res, next) => {
  try {
    const { logs } = req.body
    if (!Array.isArray(logs) || logs.length === 0) {
      return error(res, 'logs array is required')
    }

    const userAgent = req.get('user-agent')
    const device = parseDevice(userAgent)

    const enrichedLogs = logs.map((log) => ({
      ...log,
      ip: req.ip,
      userAgent,
      device,
    }))

    const result = await logsService.createBatch(enrichedLogs)
    success(res, { inserted: result.length })
  } catch (err) {
    logger.error({ err }, 'Failed to create batch logs')
    next(err)
  }
}

export const query = async (req, res, next) => {
  try {
    const { type, level, appName, userId, startTime, endTime, keyword, page, limit } = req.query

    const result = await logsService.query(
      { type, level, appName, userId, startTime, endTime, keyword },
      { page: parseInt(page) || 1, limit: parseInt(limit) || 50 }
    )

    success(res, result)
  } catch (err) {
    logger.error({ err }, 'Failed to query logs')
    next(err)
  }
}

export const getStats = async (req, res, next) => {
  try {
    const { appName } = req.query
    const stats = await logsService.getStats({ appName })
    success(res, stats)
  } catch (err) {
    logger.error({ err }, 'Failed to get log stats')
    next(err)
  }
}
