import Log from './logs.model.js'

export const create = async (logData) => {
  const log = new Log(logData)
  return log.save()
}

export const createBatch = async (logs) => {
  return Log.insertMany(logs, { ordered: false })
}

export const query = async (filters = {}, options = {}) => {
  const { page = 1, limit = 50, sort = { createdAt: -1 } } = options
  const skip = (page - 1) * limit

  const query = {}

  if (filters.type) query.type = filters.type
  if (filters.level) query.level = filters.level
  if (filters.appName) query.appName = filters.appName
  if (filters.userId) query.userId = filters.userId
  if (filters.startTime || filters.endTime) {
    query.createdAt = {}
    if (filters.startTime) query.createdAt.$gte = new Date(filters.startTime)
    if (filters.endTime) query.createdAt.$lte = new Date(filters.endTime)
  }
  if (filters.keyword) {
    query.message = { $regex: filters.keyword, $options: 'i' }
  }

  const [list, total] = await Promise.all([
    Log.find(query).sort(sort).skip(skip).limit(limit).lean(),
    Log.countDocuments(query),
  ])

  return {
    list,
    pagination: {
      page,
      limit,
      total,
      pages: Math.ceil(total / limit),
    },
  }
}

// 获取今日开始时间
const getTodayStart = () => {
  const today = new Date()
  today.setHours(0, 0, 0, 0)
  return today
}

// 获取昨日时间范围
const getYesterdayRange = () => {
  const today = getTodayStart()
  const yesterday = new Date(today)
  yesterday.setDate(yesterday.getDate() - 1)
  return { start: yesterday, end: today }
}

export const getStats = async (filters = {}) => {
  const todayStart = getTodayStart()
  const { start: yesterdayStart, end: yesterdayEnd } = getYesterdayRange()

  const baseMatch = {}
  if (filters.appName) baseMatch.appName = filters.appName

  // 并行查询所有统计数据
  const [todayTotal, yesterdayTotal, todayErrors, yesterdayErrors, todayApps, yesterdayApps, hourlyTrend, recentLogs] =
    await Promise.all([
      // 今日日志总数
      Log.countDocuments({ ...baseMatch, createdAt: { $gte: todayStart } }),
      // 昨日日志总数
      Log.countDocuments({ ...baseMatch, createdAt: { $gte: yesterdayStart, $lt: yesterdayEnd } }),
      // 今日错误日志
      Log.countDocuments({ ...baseMatch, createdAt: { $gte: todayStart }, level: { $in: ['error', 'fatal'] } }),
      // 昨日错误日志
      Log.countDocuments({
        ...baseMatch,
        createdAt: { $gte: yesterdayStart, $lt: yesterdayEnd },
        level: { $in: ['error', 'fatal'] },
      }),
      // 今日活跃应用数
      Log.distinct('appName', { ...baseMatch, createdAt: { $gte: todayStart } }),
      // 昨日活跃应用数
      Log.distinct('appName', { ...baseMatch, createdAt: { $gte: yesterdayStart, $lt: yesterdayEnd } }),
      // 按小时分组的趋势（今日）
      Log.aggregate([
        { $match: { ...baseMatch, createdAt: { $gte: todayStart } } },
        {
          $group: {
            _id: { $hour: '$createdAt' },
            count: { $sum: 1 },
          },
        },
        { $sort: { _id: 1 } },
      ]),
      // 最近日志
      Log.find(baseMatch).sort({ createdAt: -1 }).limit(10).lean(),
    ])

  // 计算变化率
  const calcChange = (today, yesterday) => {
    if (yesterday === 0) return today > 0 ? 100 : 0
    return Number((((today - yesterday) / yesterday) * 100).toFixed(1))
  }

  // 填充24小时趋势数据
  const trend = Array.from({ length: 24 }, (_, hour) => {
    const found = hourlyTrend.find((h) => h._id === hour)
    return { hour, count: found?.count || 0 }
  })

  return {
    // 今日日志
    todayTotal: {
      count: todayTotal,
      change: calcChange(todayTotal, yesterdayTotal),
    },
    // 错误日志
    errorTotal: {
      count: todayErrors,
      change: calcChange(todayErrors, yesterdayErrors),
    },
    // 活跃应用
    activeApps: {
      count: todayApps.length,
      change: todayApps.length - yesterdayApps.length,
    },
    // 按小时趋势
    trend,
    // 最近日志
    recentLogs,
  }
}
