import mongoose from 'mongoose'

// 最大存储大小（字符数）
const MAX_BODY_SIZE = 2000

// 截断大数据，保留摘要
const truncateBody = (body) => {
  if (body === null || body === undefined) return { data: null, size: 0, truncated: false }

  const str = typeof body === 'string' ? body : JSON.stringify(body)
  const size = str.length

  if (size <= MAX_BODY_SIZE) {
    return { data: body, size, truncated: false }
  }

  // 超过限制，截断并标记
  const truncatedStr = str.slice(0, MAX_BODY_SIZE)
  try {
    // 尝试保持 JSON 格式（可能不完整）
    return { data: truncatedStr + '...[truncated]', size, truncated: true }
  } catch {
    return { data: truncatedStr + '...[truncated]', size, truncated: true }
  }
}

const logSchema = new mongoose.Schema(
  {
    type: {
      type: String,
      enum: ['api', 'error', 'event', 'behavior', 'custom'],
      default: 'custom',
      index: true,
    },
    level: {
      type: String,
      enum: ['debug', 'info', 'warn', 'error', 'fatal'],
      default: 'info',
      index: true,
    },
    message: {
      type: String,
      required: true,
    },
    appName: {
      type: String,
      index: true,
    },
    userId: {
      type: String,
      index: true,
    },
    // 环境/版本
    env: {
      type: String,
      enum: ['development', 'staging', 'production'],
    },
    appVersion: String, // 应用版本号
    // 设备信息（自动从 userAgent 解析）
    device: {
      type: { type: String }, // mobile | tablet | desktop
      os: String, // iOS 17.1 | Android 14 | Windows 11
      browser: String, // Chrome 120 | Safari 17
    },
    // API 请求专用字段
    api: {
      method: String,
      url: String,
      statusCode: Number,
      duration: Number,
      request: {
        data: mongoose.Schema.Types.Mixed,
        size: Number,
        truncated: Boolean,
      },
      response: {
        data: mongoose.Schema.Types.Mixed,
        size: Number,
        truncated: Boolean,
      },
    },
    // 错误专用字段
    error: {
      name: String,
      message: String,
      stack: String,
      componentStack: String,
    },
    metadata: {
      type: mongoose.Schema.Types.Mixed,
    },
    userAgent: String,
    ip: String,
    url: String,
  },
  {
    timestamps: true,
    versionKey: false,
  }
)

// 存储前自动截断大数据
logSchema.pre('save', function (next) {
  if (this.api?.request?.data !== undefined) {
    this.api.request = truncateBody(this.api.request.data)
  }
  if (this.api?.response?.data !== undefined) {
    this.api.response = truncateBody(this.api.response.data)
  }
  next()
})

// 批量插入时也处理
logSchema.pre('insertMany', function (next, docs) {
  docs.forEach((doc) => {
    if (doc.api?.request?.data !== undefined) {
      doc.api.request = truncateBody(doc.api.request.data)
    }
    if (doc.api?.response?.data !== undefined) {
      doc.api.response = truncateBody(doc.api.response.data)
    }
  })
  next()
})

// TTL index: auto delete logs after 360 days
logSchema.index({ createdAt: 1 }, { expireAfterSeconds: 360 * 24 * 60 * 60 })

// Compound index for common queries
logSchema.index({ type: 1, appName: 1, createdAt: -1 })
logSchema.index({ type: 1, level: 1, createdAt: -1 })

export default mongoose.model('Log', logSchema)
