import { getAttrList, getConnectionList, getTagList, getEventList, getGroupList, getMaterialList } from '@/api/flow'

// 用户属性类型
export interface AttrType {
  id: string
  name: string
  type: string
  format: string
  english_name: string
  pinyin: string
}
// 事件类型
interface MetaEventType {
  id: string
  name: string
  english_name: string
  pinyin: string
  connection_name: string
  attributes: AttrType[]
}

// 标签类型
export interface TagType {
  id: string
  name: string
}

// 连接类型
export interface ConnectionType {
  id: string
  name: string
  type: number
  status: '0' | '1'
  config_attributes: any
}

// 群组类型
export interface GroupType {
  id: string
  name: string
}

// 群组类型
export interface MaterialType {
  id: string
  name: string
  content: string
  type: string
}

// 所有用户属性
const userAttrList = ref<AttrType[]>([])
export function useUserAttr() {
  function fetchUserAttrList() {
    getAttrList().then((res: any) => {
      userAttrList.value = res.data.items
    })
  }
  onMounted(() => {
    if (!userAttrList.value?.length) fetchUserAttrList()
  })

  function getUserAttr(attrId: string) {
    return userAttrList.value.find(item => item.id === attrId)
  }

  return { fetchUserAttrList, userAttrList, getUserAttr }
}

// 所有事件
const metaEventList = ref<MetaEventType[]>([])
export function useMetaEvent() {
  function fetchMetaEventList() {
    getEventList().then((res: any) => {
      metaEventList.value = res.data.items
    })
  }
  onMounted(() => {
    if (!metaEventList.value?.length) fetchMetaEventList()
  })

  function getMetaEvent(eventId: string) {
    return metaEventList.value.find(item => item.id === eventId)
  }

  return { fetchMetaEventList, metaEventList, getMetaEvent }
}

// 所有标签
const tagList = ref<TagType[]>([])
export function useTag() {
  function fetchTagList() {
    getTagList().then((res: any) => {
      tagList.value = res.data.items
    })
  }
  onMounted(() => {
    if (!tagList.value?.length) fetchTagList()
  })

  function getTag(tagId: string) {
    return tagList.value.find(item => item.id === tagId)
  }
  return { fetchTagList, tagList, getTag }
}

// 所有连接
const allConnectionList = ref<ConnectionType[]>([])
export function useConnection(type?: number) {
  const connectionType = ref<number | undefined>(type)
  function fetchConnectionList() {
    getConnectionList().then((res: any) => {
      allConnectionList.value = res.data.items.map((item: any) => {
        const attrs =
          typeof item.config_attributes === 'string' ? JSON.parse(item.config_attributes) : item.config_attributes
        const name = Array.isArray(attrs) ? attrs.find((item: any) => item.prop === 'name')?.value : attrs.name
        return { ...item, config_attributes: attrs, name }
      })
    })
  }
  const connectionList = computed(() => {
    if (connectionType.value !== undefined) {
      return allConnectionList.value.filter(item => item.type === connectionType.value)
    }
    return allConnectionList.value
  })

  onMounted(() => {
    if (!allConnectionList.value?.length) fetchConnectionList()
  })
  function getConnection(connectionId: string) {
    return allConnectionList.value.find(item => item.id === connectionId)
  }
  return { fetchConnectionList, connectionList, connectionType, getConnection }
}

// 所有群组
const groupList = ref<GroupType[]>([])
export function useGroup() {
  function fetchGroupList() {
    getGroupList().then((res: any) => {
      groupList.value = res.data.items
    })
  }
  onMounted(() => {
    if (!groupList.value?.length) fetchGroupList()
  })

  function getGroup(groupId: string) {
    return groupList.value.find(item => item.id === groupId)
  }
  return { fetchGroupList, groupList, getGroup }
}

// 所有营销资料
const allMaterialList = ref<MaterialType[]>([])
export function useMaterial(type?: string) {
  const materialType = ref<string | undefined>(type)
  function fetchMaterialList() {
    getMaterialList().then((res: any) => {
      allMaterialList.value = res.data.items
    })
  }
  const materialList = computed(() => {
    if (materialType.value !== undefined) {
      return allMaterialList.value.filter(item => item.type === materialType.value)
    }
    return allMaterialList.value
  })
  onMounted(() => {
    if (!allMaterialList.value?.length) fetchMaterialList()
  })
  return { fetchMaterialList, materialList, materialType }
}
