import {
  getMetaUserAttrList,
  getMetaEventList,
  getTagList,
  getConnectionList,
  getUserList,
  getGroupList,
} from '@/api/base'
import { useMapStore } from '@/stores/map'

// 用户属性类型
export interface AttrType {
  id: string
  name: string
  type: string
  format: string
  english_name: string
  pinyin: string
  experiment_meta_event_id?: string
}
// 事件类型
interface MetaEventType {
  id: string
  name: string
  english_name: string
  pinyin: string
  event_attrs: AttrType[]
}

// 标签类型
export interface TagType {
  id: string
  name: string
  label: string
}

// 连接类型
export interface ConnectionType {
  id: string
  name: string
  type: number
  status: '0' | '1'
  config_attributes: any
}

// 群组类型
export interface GroupType {
  id: string
  name: string
}

// 所有用户属性
const userAttrList = ref<AttrType[]>([])
const userAttrLoading = ref(false)
export function useUserAttr() {
  async function fetchUserAttrList() {
    if (userAttrLoading.value) return
    userAttrLoading.value = true
    await getMetaUserAttrList({ check_role: true }).then((res: any) => {
      userAttrList.value = res.data.items
    })
    userAttrLoading.value = false
  }
  onMounted(() => {
    if (!userAttrList.value?.length) fetchUserAttrList()
  })
  return { fetchUserAttrList, userAttrList, userAttrLoading }
}

// 所有事件
const metaEventList = ref<MetaEventType[]>([])
const metaEventLoading = ref(false)
export function useMetaEvent() {
  async function fetchMetaEventList() {
    if (metaEventLoading.value) return
    metaEventLoading.value = true
    await getMetaEventList({ check_role: true }).then((res: any) => {
      metaEventList.value = res.data.items
    })
    metaEventLoading.value = false
  }
  onMounted(() => {
    if (!metaEventList.value?.length) fetchMetaEventList()
  })
  return { fetchMetaEventList, metaEventList, metaEventLoading }
}

// 所有标签
const tagList = ref<TagType[]>([])
const tagLoading = ref(false)
export function useTag() {
  async function fetchTagList() {
    if (tagLoading.value) return
    tagLoading.value = true
    await getTagList({ check_role: 1 }).then((res: any) => {
      tagList.value = res.data.items
    })
    tagLoading.value = false
  }
  onMounted(() => {
    if (!tagList.value?.length) fetchTagList()
  })
  return { fetchTagList, tagList, tagLoading }
}

// 所有连接
const connectionList = ref<ConnectionType[]>([])
const connectionLoading = ref(false)
export function useConnection() {
  async function fetchConnectionList() {
    if (connectionLoading.value) return
    connectionLoading.value = true
    const connectionType = useMapStore().getMapValuesByKey('experiment_connection_type')
    await getConnectionList().then((res: any) => {
      connectionList.value = res.data.items.map((item: any) => {
        const connection = connectionType.find((type) => type.value == item.type)
        const attrs =
          typeof item.config_attributes === 'string' ? JSON.parse(item.config_attributes) : item.config_attributes
        const name = Array.isArray(attrs) ? attrs.find((item: any) => item.prop === 'name')?.value : attrs.name
        return { ...item, config_attributes: attrs, name, type_name: connection?.label || item.type }
      })
    })
    connectionLoading.value = false
  }
  onMounted(() => {
    if (!connectionList.value?.length) fetchConnectionList()
  })
  return { fetchConnectionList, connectionList, connectionLoading }
}

// 所有成员
export interface UserType {
  sso_id: string
  name: string
  pinyin: number
}
const userList = ref<UserType[]>([])
export function useUser() {
  const [me] = userList.value
  const userValue = ref(me?.sso_id)
  async function fetchUserList() {
    const res = await getUserList()
    let { me, students = [], teachers = [] } = res.data.items
    me = { ...me, role: 'me' }
    students = students.map((item: any) => {
      return { ...item, role: 'student' }
    })
    teachers = teachers.map((item: any) => {
      return { ...item, role: 'teacher' }
    })
    userValue.value = me.sso_id
    userList.value = [me, ...teachers, ...students]
  }
  onMounted(() => {
    if (!userList.value?.length) fetchUserList()
  })
  return { fetchUserList, userList, userValue }
}

// 所有群组
const groupList = ref<GroupType[]>([])
export function useGroup() {
  function fetchGroupList() {
    getGroupList().then((res: any) => {
      groupList.value = res.data.items
    })
  }
  onMounted(() => {
    if (!groupList.value?.length) fetchGroupList()
  })

  function getGroup(groupId: string) {
    return groupList.value.find((item) => item.id === groupId)
  }
  return { fetchGroupList, groupList, getGroup }
}
